#if NUNIT
using System;
using System.Collections;

using NUnit.Framework;

using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.DAL.HL7.Parsers;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages;
using gov.va.med.vbecs.DAL.HL7AL;

using gov.va.med.vbecs.UnitTests;

namespace gov.va.med.vbecs.DAL.HL7.UnitTests.OpenLibrary
{
	[TestFixture]
	public class HL7UtilityUnitTest : BaseTest
	{
		#region Variables

		private HL7Interface _cprsInterface;
		private HL7OmgMessage _omgMessage;

		private const string CPRS_CLINICAL_ORDER_MESSAGE = "OMG^O19";

		private const string CPRS_OMG_MESSAGE = 
			"MSH|^~\\&|OERR|589|VBECS|589|20110408142230-0500||OMG^O19|5894298218|T|2.4|||AL|AL|USA" + CARRIAGE_RETURN +
			"PID|||5890045223V539615^^^^NI~666001117^^^^SS~111111111111111112^^^^PI||VBECSLDPATIENT^SEVEN||19580419|M" + CARRIAGE_RETURN +
			"PV1||O|KC-1010 NEW PT CLINIC 2|||||||" + CARRIAGE_RETURN +
			"ORC|NW|22167068^OR||22167062^OR|||||201104081422-0500|53367^USER^TEST||53367^USER^TEST||||^Hold for OR|589^VA HEARTLAND - WEST, VISN 15|||" + CARRIAGE_RETURN +
			"OBR|1|22167068^OR||5^CRYOPRECIPITATE^99VBC^^|||||||O||1209887;3110408;7;6528||^^||||||||||||8^^^^201104081422-0500^R^^" + CARRIAGE_RETURN +
			"NTE|1||Testing Order Comments" + CARRIAGE_RETURN +
			"DG1|1||^^^^^|||A" + CARRIAGE_RETURN;

		private string _ackMessageString;

		private char [] _encodingCharacters = new char [] { '^', '~', '|', '\\', '&' };

		#endregion

		[SetUp]
		public void SetUp()
		{
			_cprsInterface = new HL7Interface( InterfaceName.CPRS.ToString() );
			//
			_omgMessage = new HL7OmgMessage( CPRS_OMG_MESSAGE );
			//
			_ackMessageString = CprsHL7Parser.ParseHL7Message( _cprsInterface, CPRS_OMG_MESSAGE ).GetMessage();
		}

		[TearDown]
		public void TearDown()
		{}

		[Test]
		public void ParseGetSegmentID_Pass_1()
		{
			string pidSegment = HL7Utility.ParseGetRequiredMessageSegment( _omgMessage.GetMessage(), SegmentTypeNames.PID );
			Assert.IsTrue( HL7Utility.ParseGetSegmentID( pidSegment ) == "PID" );
		}

		[Test]
		[ExpectedException( typeof( ArgumentNullException ) )]
		public void ParseGetSegmentID_Fail_1()
		{
			string nullException = HL7Utility.ParseGetSegmentID( null  );
		}

		[Test]
		public void ParseGetSegmentID_Fail_2()
		{
			Assert.IsTrue( HL7Utility.ParseGetSegmentID( "PI" ) == null );
		}

		[Test]
		public void ParseGetAckCode_Pass_1()
		{
			string ackCode = HL7Utility.ParseGetAckCode( _ackMessageString );
			Assert.IsTrue( (ackCode == "AA") || (ackCode == "AE") || (ackCode == "AR") );
		}

		[Test]
		[ExpectedException( typeof( ArgumentNullException ) )]
		public void ParseGetAckCode_Fail_1()
		{
			string nullException = HL7Utility.ParseGetAckCode( null );
		}

		[Test]
		[ExpectedException( typeof( HL7Exception ) )]
		public void ParseGetAckCode_Fail_2()
		{
			string argException = HL7Utility.ParseGetAckCode( CPRS_OMG_MESSAGE );
		}

		[Test]
		public void ParseGetHL7MessageTypeAndTriggerEventString_Pass_1()
		{
			Assert.IsTrue( HL7Utility.ParseGetHL7MessageTypeAndTriggerEventString( CPRS_OMG_MESSAGE ) == CPRS_CLINICAL_ORDER_MESSAGE );
		}

		[Test]
		[ExpectedException( typeof (ArgumentNullException) )]
		public void ParseGetHL7MessageTypeAndTriggerEventString_Fail_1()
		{
			string nullException = HL7Utility.ParseGetHL7MessageTypeAndTriggerEventString( null );
		}

		[Test]
		public void GetMessageControlID_Pass_1()
		{
			Assert.IsTrue( HL7Utility.GetMessageControlID( CPRS_OMG_MESSAGE ) != null );
		}

		[Test]
		[ExpectedException( typeof ( ArgumentNullException ) )]
		public void GetMessageControlID_Fail_1()
		{
			string nullException = HL7Utility.GetMessageControlID( null );
		}

		[Test]
		public void ParseGetMessageDelimiters_Pass_1()
		{
			Assert.IsTrue( HL7Utility.ParseGetMessageDelimiters( CPRS_OMG_MESSAGE ).Length == 5 ); 
		}

		[Test]
		[ExpectedException( typeof( ArgumentNullException ) )]
		public void ParseGetMessageDelimiters_Fail_1()
		{
			char[] nullException = HL7Utility.ParseGetMessageDelimiters( null );
		}

		[Test]
		[ExpectedException( typeof( ArgumentException ) )]
		public void ParseGetMessageDelimiters_Fail_2()
		{
			char[] nullException = HL7Utility.ParseGetMessageDelimiters( CPRS_OMG_MESSAGE.Remove( 0, 3 ) );
		}

		[Test]
		public void ParseGetAllMessageSegments_Pass_1()
		{
			Assert.IsTrue( HL7Utility.ParseGetAllMessageSegments( CPRS_OMG_MESSAGE ) != null );
		}

		[Test]
		[ExpectedException( typeof( ArgumentNullException ) )]
		public void ParseGetAllMessageSegments_Fail_1()
		{
			string[] nullException = HL7Utility.ParseGetAllMessageSegments( null );
		}

		[Test]
		public void ParseGetOptionalMessageSegment_Pass_1()
		{
			Assert.IsTrue( HL7Utility.ParseGetOptionalMessageSegment( CPRS_OMG_MESSAGE, SegmentTypeNames.MSH ) != null );
		}

		[Test]
		[ExpectedException( typeof( ArgumentNullException ) )]
		public void ParseGetOptionalMessageSegment_Fail_1()
		{
			string nullException = HL7Utility.ParseGetOptionalMessageSegment( null, SegmentTypeNames.MSH  );
		}

		[Test]
		public void ParseGetOptionsMessageSegment_Pass_2()
		{
			Assert.IsNull( HL7Utility.ParseGetOptionalMessageSegment( CPRS_OMG_MESSAGE, SegmentTypeNames.MRG ) );
		}

		[Test]
		public void ParseGetRequiredMessageSegment_Pass_1()
		{
			Assert.IsTrue( HL7Utility.ParseGetRequiredMessageSegment( CPRS_OMG_MESSAGE, SegmentTypeNames.MSH ) != null  ); 
		}

		[Test]
		[ExpectedException( typeof( HL7Exception ) )]
		public void ParseGetRequireMessageSegment_Fail_1()
		{
			string hl7Exception = HL7Utility.ParseGetRequiredMessageSegment( null, SegmentTypeNames.MSH );
		}

		[Test]
		[ExpectedException( typeof( HL7Exception ) )]
		public void ParseGetRequireMessageSegment_Fail_2()
		{
			string hl7Exception = HL7Utility.ParseGetRequiredMessageSegment( CPRS_OMG_MESSAGE, SegmentTypeNames.MSA ); 
		}


		[Test]
		public void ParseGetSegmentData_Pass_1()
		{
			string orcSegment = HL7Utility.ParseGetRequiredMessageSegment( CPRS_OMG_MESSAGE, SegmentTypeNames.ORC );
			Assert.IsTrue( HL7Utility.ParseGetSegmentData( orcSegment, "|", 1 ) == "NW" );
		}

		[Test]
		[ExpectedException( typeof( ArgumentNullException ) )]
		public void ParseGetSegmentData_Fail_1()
		{
			string nullException = HL7Utility.ParseGetSegmentData( null, "|", 1 );
		}

		[Test]
		[ExpectedException( typeof( ArgumentNullException ) )]
		public void ParseGetSegmentData_Fail_2()
		{
			string orcSegment = HL7Utility.ParseGetRequiredMessageSegment( CPRS_OMG_MESSAGE, SegmentTypeNames.ORC );
			string nullException = HL7Utility.ParseGetSegmentData( orcSegment, null, 1 );
		}

		[Test]
		[ExpectedException( typeof( ArgumentOutOfRangeException ) )]
		public void ParseGetSegmentData_Fail_3()
		{
			string orcSegment = HL7Utility.ParseGetRequiredMessageSegment( CPRS_OMG_MESSAGE, SegmentTypeNames.ORC );
			string argOutOfRangeException = HL7Utility.ParseGetSegmentData( orcSegment, "|", orcSegment.Length + 1 );
		}

		[Test]
		public void ParseGetStringArray_Pass_1()
		{
			Assert.IsTrue( HL7Utility.ParseGetStringArray( CPRS_OMG_MESSAGE, "|" ) != null );
		}

		[Test]
		[ExpectedException( typeof( ArgumentNullException ) )]
		public void ParseGetStringArray_Fail_1()
		{
			string[] nullException = HL7Utility.ParseGetStringArray( null, "|" );
		}

		[Test]
		[ExpectedException( typeof( ArgumentNullException ) )]
		public void ParseGetStringArray_Fail_2()
		{
			string[] nullException = HL7Utility.ParseGetStringArray( CPRS_OMG_MESSAGE, null );
		}

		[Test]
		public void ParseGetOrderComments_Pass_1()
		{
			Assert.IsTrue( HL7Utility.ParseGetOrderComments( CPRS_OMG_MESSAGE ).Count.ToString() == "1" );
		}

		[Test]
		[ExpectedException( typeof( ArgumentNullException ) )]
		public void ParseGetOrderComments_Fail_1()
		{
			ArrayList ar = HL7Utility.ParseGetOrderComments( null );
		}

		[Test]
		public void ConvertString_Pass_1()
		{
			Assert.IsTrue( HL7Utility.ConvertString( string.Empty ) == null );
		}

		[Test]
		public void ConvertString_Pass_2()
		{
			Assert.IsTrue( HL7Utility.ConvertString( "valid string" ) == "valid string" );
		}

		[Test]
		public void GetInnerExceptionFromException_Pass_1()
		{
			Assert.AreEqual( HL7Utility.GetInnerExceptionFromException( new Exception("Exception Unit Test")), "Exception Unit Test\n" );
		}

		[Test]
		[ExpectedException( typeof( ArgumentNullException ) )]
		public void GetInnerExceptionFromException_Fail_1()
		{
			HL7Utility.GetInnerExceptionFromException(null);
		}

		[Test]
		public void GetInnerExceptionFromException_Pass_2()
		{
			Assert.AreEqual( HL7Utility.GetInnerExceptionFromException( new Exception("Exception Unit Test", new Exception("Inner Exception Text"))), "Exception Unit Test\nInner Exception Text\n" );

		}

		[Test]
		public void ParseGetErrorCodeAndMessage_Pass()
		{
			string hl7Message = string.Concat(
				"MSH^~|\\&^VBECS^VBECS^WBC^589^20080526032505-0500^^BRT~O32^VBECS8052603208125^T^2.5",
				CARRIAGE_RETURN,
				"MSA^AE^WBC200805261513234027",
				CARRIAGE_RETURN,
				"ERR^^^206~You locked this record in another area 05/22/2008 15:17.^E",
				CARRIAGE_RETURN );
			//
			string [] errorCodeAndMessage = HL7.OpenLibrary.HL7Utility.ParseGetErrorCodeAndMessage( hl7Message );
			//
			Assert.IsNotNull( errorCodeAndMessage );
			//
			Assert.IsNotNull( errorCodeAndMessage[0] );
			//
			Assert.IsNotNull( errorCodeAndMessage[1] );
		}

		[Test]
		[ExpectedException( typeof( ArgumentNullException ) )]
		public void ParseGetErrorCodeAndMessage_Fail()
		{
			Assert.IsNotNull( HL7.OpenLibrary.HL7Utility.ParseGetErrorCodeAndMessage( null ) );
		}

		[Test]
		public void ParseGetVistAPatientId_Pass()
		{
			string pid = "PID^1^1000055397V547770^412618^^VBPATIENT~SIXTY^^19450101^M^^^^^^^^^^^60^^^^^^^^^^";
			//
			long dfn = HL7.OpenLibrary.HL7Utility.ParseGetVistAPatientId( pid, _encodingCharacters );
			//
			Assert.IsTrue( dfn > 0 );
		}

		[Test]
		[ExpectedException( typeof( ArgumentNullException ) )]
		public void ParseGetVistAPatientId_Fail_1()
		{
			string pid = "PID^1^1000055397V547770^412618^^VBPATIENT~SIXTY^^19450101^M^^^^^^^^^^^60^^^^^^^^^^";
			// 
			long dfn = HL7.OpenLibrary.HL7Utility.ParseGetVistAPatientId(pid, null);
			//
			Assert.IsTrue( dfn > 0 );
		}

		[Test]
		[ExpectedException( typeof( ArgumentNullException ) )]
		public void ParseGetVistAPatientId_Fail_2()
		{
			long dfn = HL7.OpenLibrary.HL7Utility.ParseGetVistAPatientId( null, _encodingCharacters );
			//
			Assert.IsTrue( dfn > 0 );
		}

		[Test]
		public void GetAckErrorCodeFromEnum_Pass()
		{
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetAckErrorCodeFromEnum(AckErrorCodes.MessageAccepted)				== 0 );
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetAckErrorCodeFromEnum(AckErrorCodes.SegmentSequenceError)		== 100 );
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetAckErrorCodeFromEnum(AckErrorCodes.RequiredFieldMissing)		== 101 );
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetAckErrorCodeFromEnum(AckErrorCodes.DataTypeError)				== 102 );
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetAckErrorCodeFromEnum(AckErrorCodes.TableValueNotFound)			== 103 );
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetAckErrorCodeFromEnum(AckErrorCodes.UnsupportedMessageType)		== 200 );
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetAckErrorCodeFromEnum(AckErrorCodes.UnsupportedEventCode)		== 201 );
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetAckErrorCodeFromEnum(AckErrorCodes.UnsupportedProcessingId)		== 202 );
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetAckErrorCodeFromEnum(AckErrorCodes.UnsupportedVersionId)		== 203 );
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetAckErrorCodeFromEnum(AckErrorCodes.UnknownKeyIdentifier)		== 204 );
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetAckErrorCodeFromEnum(AckErrorCodes.DuplicateKeyIdentifier)		== 205 );
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetAckErrorCodeFromEnum(AckErrorCodes.ApplicationRecordLocked)		== 206 );
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetAckErrorCodeFromEnum(AckErrorCodes.ApplicationInternalError)	== 207 );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetAckErrorCodeFromEnum_Fail()
		{
			
		}

		[Test]
		public void GetAckErrorSeverityFromEnum_Pass()
		{
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetAckErrorSeverityFromEnum(AckErrorSeverity.Error )		== 'E' );
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetAckErrorSeverityFromEnum(AckErrorSeverity.Information)	== 'I' );
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetAckErrorSeverityFromEnum(AckErrorSeverity.Warning)		== 'W' );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetAckErrorSeverityFromEnum_Fail()
		{
		}

		[Test]
		public void GetBloodProductCommentTypeEnumFromString_Pass()
		{
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetBloodProductCommentTypeEnumFromString("VA-BCR")	== BloodProductCommentType.Barcode );
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetBloodProductCommentTypeEnumFromString("RE")		== BloodProductCommentType.FreeText );
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetBloodProductCommentTypeEnumFromString("ZZ")		== BloodProductCommentType.Unknown );
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetBloodProductCommentTypeEnumFromString(null)		== BloodProductCommentType.Unknown );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetBloodProductCommentTypeEnumFromString_Fail()
		{
		}

		[Test]
		public void GetBloodProductMessageStatusEnumFromChar_Pass()
		{
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetBloodProductMessageStatusEnumFromChar('F')	== BloodProductMessageStatus.Final );
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetBloodProductMessageStatusEnumFromChar('P')	== BloodProductMessageStatus.Preliminary );
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetBloodProductMessageStatusEnumFromChar('Z')	== BloodProductMessageStatus.Unknown );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetBloodProductMessageStatusEnumFromChar_Fail()
		{
		}

		[Test]
		public void GetBloodProductTransfusionStatusEnumFromString_Pass()
		{
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetBloodProductTransfusionStatusEnumFromString("RA")	== BloodProductTransfusionStatus.RA );
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetBloodProductTransfusionStatusEnumFromString("RL")	== BloodProductTransfusionStatus.RL );
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetBloodProductTransfusionStatusEnumFromString("TR")	== BloodProductTransfusionStatus.TR );
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetBloodProductTransfusionStatusEnumFromString("TX")	== BloodProductTransfusionStatus.TX );
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetBloodProductTransfusionStatusEnumFromString("WA")	== BloodProductTransfusionStatus.WA );
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetBloodProductTransfusionStatusEnumFromString("ZZ")	== BloodProductTransfusionStatus.UK );
			Assert.IsTrue( HL7.OpenLibrary.HL7Utility.GetBloodProductTransfusionStatusEnumFromString(null)	== BloodProductTransfusionStatus.UK );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetBloodProductTransfusionStatusEnumFromString_Fail()
		{
		}
	}
}
#endif
